/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/***
 * \file: Dm8127_Platform.c
 *
 * \brief: source file for Dm8127 Configuration.
 *
 * \b \Descr: This file initialize the Platform like UART and the Interfaces
 * \n         configuration are available in the DM8127 Board.
 * \n         This file write as alternate of the GEL file for DM8127 processor.
 *
 */

#include "dm8127_types.h"
#include "dm8127_platform.h"
#include "dm8127_evm.h"
#include <stdarg.h>
#include "uart.h"

unsigned int gu8AdpllClkInit;
int CLKIN ;
int HSMODE;
#if 0
VOID I2C0Enable(VOID)
{
//    UARTClkEnable();
    WR_MEM_32(CM_ALWON_I2C_0_CLKSTCTRL,    0x2); /*Enable I2C0 Clock*/
    while(RD_MEM_32(CM_ALWON_I2C_0_CLKSTCTRL)!=0x2); /*Poll for Module is functional*/

    WR_MEM_32(PINCNTL263, 0x00060001);
    WR_MEM_32(PINCNTL264, 0x00060001);

    Uart_stringSend("\t I2C0 is initialized \n");
}

VOID I2C1Enable(VOID)
{
    WR_MEM_32(PINCNTL78, 0x00060002);
    WR_MEM_32(PINCNTL79, 0x00060002);

    WR_MEM_32(CM_ALWON_I2C_1_CLKSTCTRL,    0x2); /*Enable I2C1 Clock*/
    while(RD_MEM_32(CM_ALWON_I2C_1_CLKSTCTRL)!=0x2);        /*Poll for Module is functional*/

    Uart_stringSend("\tI2C1 is initialized \n");
}

VOID I2C2Enable(VOID)
{
	WR_MEM_32(PINCNTL135, 0x000E0040);
    WR_MEM_32(PINCNTL136, 0x000E0040);
    WR_MEM_32(CM_ALWON_I2C_0_CLKSTCTRL,    0x2); /*Enable I2C2 Clock*/
    while(RD_MEM_32(CM_ALWON_I2C_0_CLKSTCTRL)!=0x2);        /*Poll for Module is functional*/

    Uart_stringSend("\t I2C2 is initialized \n");
}



/* GPIO clocks enable */
void GPIO0ClkEnable(void)
{
   WR_MEM_32(CM_ALWON_GPIO_0_CLKCTRL, GPIO_EN);
    while(RD_MEM_32(CM_ALWON_GPIO_0_CLKCTRL)!= GPIO_EN);
}

void GPIO1ClkEnable(void)
{
   WR_MEM_32(CM_ALWON_GPIO_1_CLKCTRL, GPIO_EN);
   while(RD_MEM_32(CM_ALWON_GPIO_1_CLKCTRL)!= GPIO_EN);
}

void GPIO2ClkEnable(void)
{
   WR_MEM_32(CM_ALWON_GPIO_2_CLKCTRL, GPIO_EN);
    while(RD_MEM_32(CM_ALWON_GPIO_2_CLKCTRL)!= GPIO_EN);
}


STATUS DCOCLK_COMP(int CLKIN,int N, int M)
{
	 	int DCOCLK;
		DCOCLK = (CLKIN/(N+1))*M;

		if(DCOCLK >= 500 && DCOCLK < 1000){
				HSMODE = 2;  //HS2 Mode
		}
		else if(DCOCLK >= 1000 && DCOCLK < 2000){
				HSMODE = 1;  //HS1 Mode
		}
		else HSMODE = 0;  //wrong configuration

		//return HSMODE;
}
VOID PLL_Clocks_Config
(
	VUINT32 Base_Address,
	UINT32 CLKIN,
	UINT32 N,
	UINT32 M,
	UINT32 M2,
	UINT32 CLKCTRL_VAL
)
{
	/* Local variables */
    UINT32 m2nval,mn2val, read_clkctrl, clk_out, ref_clk, clkout_dco = INIT_VAL;

    m2nval = (M2<<16) | N;
    mn2val =  M;
#if 1
	ref_clk     = CLKIN/(N+1);
    clkout_dco  = ref_clk*M;
    clk_out     = clkout_dco/M2;
#endif

    WR_MEM_32((VUINT32 *)(Base_Address + CLKCTRL), (RD_MEM_32(VUINT32 *)(Base_Address + CLKCTRL)| 0x00800000));
	while (( (RD_MEM_32((VUINT32 *)(Base_Address + ADPLL_STATUS))) & 0x00000101) != 0x00000101);

    WR_MEM_32(Base_Address + CLKCTRL, RD_MEM_32(Base_Address + CLKCTRL)& 0xfffffffe);
	delay(3);

	WR_MEM_32((VUINT32 *)(Base_Address + M2NDIV    ),m2nval);
	WR_MEM_32((VUINT32 *)(Base_Address + MN2DIV    ),mn2val);
	delay(3);

	WR_MEM_32((VUINT32 *)(Base_Address+TENABLEDIV),0x1);
	delay(3);

	WR_MEM_32((VUINT32 *)(Base_Address+TENABLEDIV),0x0);
	delay(3);

	WR_MEM_32((VUINT32 *)(Base_Address+TENABLE   ),0x1);
	delay(3);

	WR_MEM_32((VUINT32 *)(Base_Address+TENABLE   ),0x0);
	delay(3);

	read_clkctrl = RD_MEM_32((VUINT32 *)Base_Address+CLKCTRL);

	/* configure the TINITZ(bit0) and CLKDCO BITS IF REQUIRED */
	WR_MEM_32((VUINT32 *)(Base_Address+CLKCTRL),(read_clkctrl & 0xff7fe3ff) | CLKCTRL_VAL);
	read_clkctrl = RD_MEM_32(Base_Address+CLKCTRL);

	/* poll for the freq,phase lock to occur */
	while (( (RD_MEM_32((VUINT32 *)(Base_Address + ADPLL_STATUS))) & 0x00000600) != 0x00000600);

	/* wait fot the clocks to get stabized */
	delay(10);

#ifdef DEBUG

	printf("clock out value %d\n",clk_out);
//    CLKOUT    = clk_out;
#endif
}




STATUS EMACClkEnable(VOID)
{
	Uart_stringSend("\tCONFIGURE PRCM CLOCKS of EMAC in progress \n\r");

    WR_MEM_32(MC_ALWON_ETHNET_CLKSTCTRL,    0x2); /* Enable Ethernet Clock*/
    while((RD_MEM_32(MC_ALWON_ETHNET_CLKSTCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/

    WR_MEM_32(MC_ALWON_ETH_0_CLKSTCTRL,    0x2); /*Enable Ethernet Clock*/
    while((RD_MEM_32(MC_ALWON_ETH_0_CLKSTCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/

    WR_MEM_32(MC_ALWON_ETH_1_CLKSTCTRL,    0x2); /*Enable Ethernet Clock*/
    while((RD_MEM_32(MC_ALWON_ETH_1_CLKSTCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/

    while(RD_MEM_32(MC_ALWON_ETHNET_CLKSTCTRL)!=0x302);        /*Poll for Module is functional*/

    Uart_stringSend("\tPRCM CLOCKS of EMAC  is complete \n\r");
}

MODENA_PLL_Config(UINT32 CLKIN,UINT32 N,UINT32 M,UINT32 M2)
{
   UINT32 rval_ctrl,ref_clk,clk_out = 0;
   UINT32 m2nval,mn2val = 0;
	ref_clk     = CLKIN/(N+1);
   clk_out     = (ref_clk*M)/M2;

	m2nval = (M2<<16) | N; //need to check the bit position of M2
   mn2val =  M;
   WR_MEM_32(MODENAPLL_M2NDIV      ,m2nval);
   WR_MEM_32(MODENAPLL_MN2DIV      ,mn2val);
	WR_MEM_32(MODENAPLL_TENABLEDIV  ,0x1);
	delay(3);
   WR_MEM_32(MODENAPLL_TENABLEDIV  ,0x0);
	delay(3);
   WR_MEM_32(MODENAPLL_TENABLE     ,0x1);
	delay(3);
   WR_MEM_32(MODENAPLL_TENABLE     ,0x0);
	delay(3);
   rval_ctrl = RD_MEM_32(MODENAPLL_CLKCTRL);
   WR_MEM_32(MODENAPLL_CLKCTRL,(rval_ctrl & 0xff7fffff) | 0x1);
   while (( (RD_MEM_32(MODENAPLL_STATUS)) & 0x00000600) != 0x00000600);
	delay(10);
	CLKOUT = clk_out;

}
cmdMPUPLL(int CLKIN,int N, int M, int M2)
	{
	    MODENA_PLL_Config(CLKIN,N,M,M2);
	    platform_write("\t MODENA ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
	}

 cmdL3PLL(int CLKIN,int N, int M, int M2)
 	{
 	    DCOCLK_COMP(CLKIN,N,M);
 	    if(HSMODE == 2){
		PLL_Clocks_Config(L3_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
		platform_write("\t L3  ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
       }
	    else if (HSMODE == 1){
	      PLL_Clocks_Config(L3_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
	      platform_write("\t L3  ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
	    }
	    else {
	    	Uart_stringSend("\t L3 PLL NOT Configured.Wrong DCOCLK Output\n");
		}

	}
cmdDSPPLL(int CLKIN,int N, int M, int M2)
	{
		DCOCLK_COMP(CLKIN,N,M);
		if(HSMODE == 2){
			PLL_Clocks_Config(DSP_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
			platform_write("\t DSP ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
		}
	    else if (HSMODE == 1){
			PLL_Clocks_Config(DSP_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
			platform_write("\t DSP ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
       }
       else{
    	   Uart_stringSend("\t DSP PLL NOT Configured.Wrong DCOCLK Output\n");
	    }

	}
cmdDSSPLL(int CLKIN, int N, int M, int M2)
	{
	    DCOCLK_COMP(CLKIN,N,M);
		if(HSMODE == 2){
			PLL_Clocks_Config(DSS_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
			platform_write("\t DSS ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
		}
	    else if (HSMODE == 1){
			PLL_Clocks_Config(DSS_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
			platform_write("\t DSS ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
       }
	    else {
	    	Uart_stringSend("\t DSS PLL NOT Configured.Wrong DCOCLK Output\n");
	    }

	}

cmdSGXPLL(int CLKIN, int N, int M, int M2)
{
		DCOCLK_COMP(CLKIN,N,M);
		if(HSMODE == 2){
			PLL_Clocks_Config(SGX_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
			platform_write("\t SGX ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
		}
	    else if (HSMODE == 1){
			PLL_Clocks_Config(SGX_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
			platform_write("\t SGX ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
       }
	    else {
	    	Uart_stringSend("\t SGX PLL NOT Configured.Wrong DCOCLK Output\n");
	    }

}
cmdIVAPLL(int CLKIN,int N, int M, int M2)
{
	    DCOCLK_COMP(CLKIN,N,M);
		if(HSMODE == 2){
			PLL_Clocks_Config(IVA_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
			platform_write("\t IVA ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
		}
	    else if (HSMODE == 1){
			PLL_Clocks_Config(IVA_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
			platform_write("\t IVA ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
       }
	    else {
	    	Uart_stringSend("\t IVA PLL NOT Configured.Wrong DCOCLK Output\n");
	    }

}
 cmdISSPLL(int CLKIN,int N, int M, int M2)
{
		DCOCLK_COMP(CLKIN,N,M);
		if(HSMODE == 2){
			PLL_Clocks_Config(ISS_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
			platform_write("\t ISS ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
		}
	    else if (HSMODE == 1){
			PLL_Clocks_Config(ISS_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
			platform_write("\t ISS ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
       }
	    else {
	    	Uart_stringSend("\t ISS PLL NOT Configured.Wrong DCOCLK Output\n");
	    }

}
// enabling the CLKOUTDCO,CLKDCOLDO
 cmdUSBPLL(int CLKIN,int N, int M, int M2)
{
		DCOCLK_COMP(CLKIN,N,M);
		if(HSMODE == 2){
			PLL_Clocks_Config(USB_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2+ADPLLJ_CLKCRTL_CLKDCO);
			platform_write("\t USB ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
		}
	    else if (HSMODE == 1){
			PLL_Clocks_Config(USB_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1+ADPLLJ_CLKCRTL_CLKDCO);
			platform_write("\t USB ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
       }
	    else {
	    	Uart_stringSend("\t USB PLL NOT Configured.Wrong DCOCLK Output\n");
	    }

}
 cmdVIDEO0PLL(int CLKIN,int N, int M, int M2)
{
		DCOCLK_COMP(CLKIN,N,M);
		if(HSMODE == 2){
			PLL_Clocks_Config(VIDEO_0_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
			platform_write("\t VIDEO-0 ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
		}
	    else if (HSMODE == 1){
			PLL_Clocks_Config(VIDEO_0_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
			platform_write("\t VIDEO-0 ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
       }
	    else {
	    	Uart_stringSend("\t VIDEO-0 PLL NOT Configured.Wrong DCOCLK Output\n");
	    }


}
 cmdVIDEO1PLL(int CLKIN,int N, int M, int M2)
{
		DCOCLK_COMP(CLKIN,N,M);
		if(HSMODE == 2){
			PLL_Clocks_Config(VIDEO_1_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
			platform_write("\t VIDEO-1 ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
		}
	    else if (HSMODE == 1){
			PLL_Clocks_Config(VIDEO_1_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
			platform_write("\t VIDEO-1 ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
       }
	    else {
	    	Uart_stringSend("\t VIDEO-1 PLL NOT Configured.Wrong DCOCLK Output\n");
	    }

}

 cmdHDMIPLL(int CLKIN,int N, int M, int M2)
{
		DCOCLK_COMP(CLKIN,N,M);
		if(HSMODE == 2){
			PLL_Clocks_Config(HDMI_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2+ADPLLJ_CLKCRTL_CLKDCO);
			platform_write("\t VIDEO-2/HDMI ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
		}
	    else if (HSMODE == 1){
			PLL_Clocks_Config(HDMI_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1+ADPLLJ_CLKCRTL_CLKDCO);
			platform_write("\t VIDEO-2/HDMI ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
       }
	    else {
	    	Uart_stringSend("\t VIDEO-2/HDMI PLL NOT Configured.Wrong DCOCLK Output\n");
	    }

}
 cmdDDRPLL(int CLKIN,int N, int M, int M2)
{
		DCOCLK_COMP(CLKIN,N,M);
		if(HSMODE == 2){
			PLL_Clocks_Config(DDR_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
			platform_write("\t DDR ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
		}
	    else if (HSMODE == 1){
			PLL_Clocks_Config(DDR_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
			platform_write("\t DDR ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
       }
	    else {
	    	Uart_stringSend("\t DDR PLL NOT Configured.Wrong DCOCLK Output\n");
	    }


}
 cmdAUDIOPLL(int CLKIN,int N, int M, int M2)
{
		DCOCLK_COMP(CLKIN,N,M);
		if(HSMODE == 2){
			PLL_Clocks_Config(AUDIO_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS2);
			platform_write("\t AUDIO ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
		}
	    else if (HSMODE == 1){
			PLL_Clocks_Config(AUDIO_PLL_BASE,CLKIN,N,M,M2,ADPLLJ_CLKCRTL_HS1);
			platform_write("\t AUDIO ADPLLJ CLKOUT  value is  = %d \n",CLKOUT);
       }
	    else {
	    	Uart_stringSend("\t AUDIO PLL NOT Configured.Wrong DCOCLK Output\n");
	    }

}



 cmdSATAPLL()
 {
	 Uart_stringSend("\t **** SATA PLL INIT IS In Progress Please wait ..... \n");
   WR_MEM_32(SATA_PLLCFG0, 0x00000004);	//cfgpll0
   delay(35);
   WR_MEM_32(SATA_PLLCFG1, 0xC12C003C);	//cfgpll1
   WR_MEM_32(SATA_PLLCFG3, 0x004008E0);	//cfgpll3
   delay(850);
    //wait for bias to be stable --50us
   WR_MEM_32(SATA_PLLCFG0, 0x00000014);	//cfgpll0
   delay(850);
   WR_MEM_32(SATA_PLLCFG0, 0x00000016);	//cfgpll0
   delay(60);
   WR_MEM_32(SATA_PLLCFG0, 0xC0000016);	//cfgpll0 -----why 2 times???????
   delay(2000);
   WR_MEM_32(SATA_PLLCFG0, 0xC0000017);	//cfgpll0
  //poll the status field to check if pll lock occured.
   while ((RD_MEM_32(SATA_PLLSTATUS) & 0x1) != 0x1);
   Uart_stringSend("\t **** SATA PLL INIT IS In DONE ***************** \n");

 }

STATUS PLL_SETUP
(
    VOID
)
{
	 cmdMPUPLL(CLKIN,1, 60 ,1);
	 cmdL3PLL (CLKIN,19,800,4);

	 cmdDSPPLL(CLKIN,19, 500, 1);
	 cmdDSSPLL(CLKIN,19, 800, 4);
	 cmdISSPLL(CLKIN,19, 800 ,4);
	 cmdIVAPLL(CLKIN,19, 532, 2);
	 cmdSGXPLL(CLKIN,19, 800, 4);
	 cmdUSBPLL(CLKIN,19,960,5);
	 cmdVIDEO0PLL(CLKIN,19, 540,10);
	 cmdVIDEO1PLL(CLKIN,19, 594,4);
	 cmdHDMIPLL(CLKIN,19, 1485,10);
	 cmdDDRPLL(CLKIN,19,DDR_FREQ, 2);
	 cmdAUDIOPLL(CLKIN,19,800,4);
	 cmdSATAPLL();

}


STATUS ALL_ADPLL_CLOCKS_ENABLE_API()
{

   if (((RD_MEM_32(DSS_PLL_BASE + ADPLL_STATUS)) & 0x00000600) == 0x00000600)
	   gu8AdpllClkInit = 1;
   else
	   gu8AdpllClkInit = 0;

   if (gu8AdpllClkInit == 0)
   {
	   Uart_stringSend("\t ****  DM8127 ALL ADPLL INIT IS In Progress ********* \n");
	   PLL_SETUP();
	   EMACClkEnable();
	   gu8AdpllClkInit = 1;
	   Uart_stringSend("\t ****  DM8127 ALL ADPLL INIT IS  Done ************** \n");
   }

}


STATUS BT_init
(

)
{

}

STATUS SD_MMC_config()
{
	Uart_stringSend("\t ***** This GEL function is not yet tested....***** \n");
		 GPIO1ClkEnable();
		WR_MEM_32(PINCNTL1, 0x00050001);
		WR_MEM_32(PINCNTL2, 0x00060001);
		WR_MEM_32(PINCNTL3, 0x00060001);
		WR_MEM_32(PINCNTL4, 0x00060001);
		WR_MEM_32(PINCNTL5, 0x00060001);
		WR_MEM_32(PINCNTL6, 0x00060001);

		WR_MEM_32(CM_ALWON_MMC_HS_1_CLKSTCTRL,    0x2); /*Enable UART0 Clock*/
		while(RD_MEM_32(CM_ALWON_MMC_HS_1_CLKSTCTRL)!=0x2);		/*Poll for Module is functional*/

		Uart_stringSend("\t ***** MMC/SD -1 is initialized....***** \n");
}


STATUS NAND_config
(
     VOID
)
{
		Uart_stringSend("\tGPMC PINMUX configuration for NAND in progress, Please wait.....  \n");
//	    ALL_ADPLL_CLOCKS_ENABLE_API();
//	    GPMCClkEnable();
	    WR_MEM_32(CM_ALWON_GPMC_CLKCTRL,    0x2); /*Enable GPMC Clock*/
	    while(RD_MEM_32(CM_ALWON_GPMC_CLKCTRL)!=0x2);
	    GPIO1ClkEnable();

	    WR_MEM_32(PINCNTL235,   RD_MEM_32(PINCNTL235)  | 0x1);
	    WR_MEM_32(PINCNTL243,   RD_MEM_32(PINCNTL243)  | 0x10);
	    WR_MEM_32(PINCNTL244,   RD_MEM_32(PINCNTL244)  | 0x10);
	    WR_MEM_32(PINCNTL245,   RD_MEM_32(PINCNTL245)  | 0x10);
	    WR_MEM_32(PINCNTL246,   RD_MEM_32(PINCNTL246)  | 0x10);
	    WR_MEM_32(PINCNTL247,   RD_MEM_32(PINCNTL247)  | 0x10);
	    WR_MEM_32(PINCNTL248,   RD_MEM_32(PINCNTL248)  | 0x10);
	    WR_MEM_32(PINCNTL249,   RD_MEM_32(PINCNTL249)  | 0x10);
	    WR_MEM_32(PINCNTL250,   RD_MEM_32(PINCNTL250)  | 0x10);
	    WR_MEM_32(PINCNTL251,   RD_MEM_32(PINCNTL251)  | 0x10);
	    WR_MEM_32(PINCNTL252,   RD_MEM_32(PINCNTL252)  | 0x10);
	    WR_MEM_32(PINCNTL253,   RD_MEM_32(PINCNTL253)  | 0x10);
	    WR_MEM_32(PINCNTL254,   RD_MEM_32(PINCNTL254)  | 0x10);
	    WR_MEM_32(PINCNTL255,   RD_MEM_32(PINCNTL255)  | 0x10);
	    WR_MEM_32(PINCNTL256,   RD_MEM_32(PINCNTL256)  | 0x10);
	    WR_MEM_32(PINCNTL257,   RD_MEM_32(PINCNTL257)  | 0x10);
	    WR_MEM_32(PINCNTL258,   RD_MEM_32(PINCNTL258)  | 0x10);
	    WR_MEM_32(PINCNTL89,    RD_MEM_32(PINCNTL89)   | 0x01);
	    WR_MEM_32(PINCNTL90,    RD_MEM_32(PINCNTL90)   | 0x01);
	    WR_MEM_32(PINCNTL91,    RD_MEM_32(PINCNTL91)   | 0x01);
	    WR_MEM_32(PINCNTL92,    RD_MEM_32(PINCNTL92)   | 0x01);
	    WR_MEM_32(PINCNTL93,    RD_MEM_32(PINCNTL93)   | 0x01);
	    WR_MEM_32(PINCNTL94,    RD_MEM_32(PINCNTL94)   | 0x01);
	    WR_MEM_32(PINCNTL95,    RD_MEM_32(PINCNTL95)   | 0x01);
	    WR_MEM_32(PINCNTL96,    RD_MEM_32(PINCNTL96)   | 0x01);
	    WR_MEM_32(PINCNTL97,    RD_MEM_32(PINCNTL97)   | 0x01);
	    WR_MEM_32(PINCNTL98,    RD_MEM_32(PINCNTL98)   | 0x01);
	    WR_MEM_32(PINCNTL99,    RD_MEM_32(PINCNTL99)   | 0x01);
	    WR_MEM_32(PINCNTL100,   RD_MEM_32(PINCNTL100)  | 0x01);
	    WR_MEM_32(PINCNTL101,   RD_MEM_32(PINCNTL101)  | 0x01);
	    WR_MEM_32(PINCNTL102,   RD_MEM_32(PINCNTL102)  | 0x01);
	    WR_MEM_32(PINCNTL103,   RD_MEM_32(PINCNTL103)  | 0x01);
	    WR_MEM_32(PINCNTL104,   RD_MEM_32(PINCNTL104)  | 0x01);
	    WR_MEM_32(PINCNTL105,   RD_MEM_32(PINCNTL105)  | 0x01);
	    WR_MEM_32(PINCNTL106,   RD_MEM_32(PINCNTL106)  | 0x01);
	    WR_MEM_32(PINCNTL107,   RD_MEM_32(PINCNTL107)  | 0x01);
	    WR_MEM_32(PINCNTL108,   RD_MEM_32(PINCNTL108)  | 0x01);
	    WR_MEM_32(PINCNTL109,   RD_MEM_32(PINCNTL109)  | 0x01);
	    WR_MEM_32(PINCNTL110,   RD_MEM_32(PINCNTL110)  | 0x01);
	    WR_MEM_32(PINCNTL111,   RD_MEM_32(PINCNTL111)  | 0x01);
	    WR_MEM_32(PINCNTL112,   RD_MEM_32(PINCNTL112)  | 0x01);
	    WR_MEM_32(PINCNTL122,   RD_MEM_32(PINCNTL122)  | 0x01);
	    WR_MEM_32(PINCNTL123,   RD_MEM_32(PINCNTL123)  | 0x01);
	    WR_MEM_32(PINCNTL124,   RD_MEM_32(PINCNTL124)  | 0x01);
	    WR_MEM_32(PINCNTL125,   RD_MEM_32(PINCNTL125)  | 0x01);
	    WR_MEM_32(PINCNTL126,   RD_MEM_32(PINCNTL126)  | 0x01);
	    WR_MEM_32(PINCNTL127,   RD_MEM_32(PINCNTL127)  | 0x01);
	    WR_MEM_32(PINCNTL128,   RD_MEM_32(PINCNTL128)  | 0x01);
	    WR_MEM_32(PINCNTL129,   RD_MEM_32(PINCNTL129)  | 0x01);
	    WR_MEM_32(PINCNTL130,   RD_MEM_32(PINCNTL130)  | 0x01);
	    WR_MEM_32(PINCNTL131,   RD_MEM_32(PINCNTL131)  | 0x01);
	    WR_MEM_32(PINCNTL132,   RD_MEM_32(PINCNTL132)  | 0x01);
	    WR_MEM_32(PINCNTL133,   RD_MEM_32(PINCNTL133)  | 0x01);
		WR_MEM_32(PINCNTL65,    RD_MEM_32(PINCNTL65)   | 0x80); /* WR_PROTECT__EN*/

	    WR_MEM_32(0x50000060,    RD_MEM_32(0x50000060) & 0xFFFFFCFF);// Bit 9:8 make 00, Non multiplex
	    WR_MEM_32(0x50000078,    RD_MEM_32(0x50000078) & 0xFFFFFFFC);
	    WR_MEM_32(0x50000078,    RD_MEM_32(0x50000078) | 2); // Map to 0x02000000 address
		WR_MEM_32(MLBP_DAT_IO_CTRL,0x00000005);

    	Uart_stringSend("\tGPMC initialization complete...\n");
}


STATUS EthernetInit(VOID)
{
	Uart_stringSend("\r\nConfiguring GMII Clk and Mux....*****");
    PLL_SETUP();
    Uart_stringSend("\r\nCONFIGURE PRCM CLOCKS of EMAC in progress \n\r");
//
//     WR_MEM_32(MC_ALWON_ETHNET_CLKSTCTRL,    0x2); /* Enable Ethernet Clock*/
//     while((RD_MEM_32(MC_ALWON_ETHNET_CLKSTCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/
//
//     WR_MEM_32(MC_ALWON_ETH_0_CLKSTCTRL,    0x2); /*Enable Ethernet Clock*/
//     while((RD_MEM_32(MC_ALWON_ETH_0_CLKSTCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/
//
//     WR_MEM_32(MC_ALWON_ETH_1_CLKSTCTRL,    0x2); /*Enable Ethernet Clock*/
//     while((RD_MEM_32(MC_ALWON_ETH_1_CLKSTCTRL) & 0x0F) !=0x2);        /*Poll for Module is functional*/
//
//     while(RD_MEM_32(MC_ALWON_ETHNET_CLKSTCTRL)!=0x302);        /*Poll for Module is functional*/
//     Uart_stringSend("PRCM CLOCKS of EMAC  is complete \n\r");

    EMACClkEnable();
	GPIO0ClkEnable();
	GPIO1ClkEnable();
	GPIO2ClkEnable();
    //I2C0Enable();
	WR_MEM_32(SERDES_REFCLK_CTL,0x00000002); //PowerDown 0x00000000-DM8127
    WR_MEM_32(GMII_SELECT, 0x00000000);

    /* PIN MUX for EMAC0 */
    WR_MEM_32(PINCNTL232, 0x00040001); /*NQ*/
    WR_MEM_32(PINCNTL233, 0x000A0001);
    WR_MEM_32(PINCNTL234, 0x000E0001);
	WR_MEM_32(PINCNTL235, 0x000C0001);
    WR_MEM_32(PINCNTL236, 0x000C0001);
    WR_MEM_32(PINCNTL237, 0x000C0001);
    WR_MEM_32(PINCNTL238, 0x000C0001);
    WR_MEM_32(PINCNTL239, 0x00040001);
    WR_MEM_32(PINCNTL240, 0x00040001);
    WR_MEM_32(PINCNTL241, 0x00040001);
    WR_MEM_32(PINCNTL242, 0x00040001);
    WR_MEM_32(PINCNTL243, 0x00040001);
    WR_MEM_32(PINCNTL244, 0x00040001);
    WR_MEM_32(PINCNTL245, 0x00040001);
    WR_MEM_32(PINCNTL246, 0x00040001);
    WR_MEM_32(PINCNTL247, 0x00040001);
    WR_MEM_32(PINCNTL248, 0x00040001);
    WR_MEM_32(PINCNTL249, 0x00000001);
    WR_MEM_32(PINCNTL250, 0x00000001);
    WR_MEM_32(PINCNTL251, 0x00000001);
    WR_MEM_32(PINCNTL252, 0x00000001);
    WR_MEM_32(PINCNTL253, 0x00000001);
    WR_MEM_32(PINCNTL254, 0x00000001);
    WR_MEM_32(PINCNTL255, 0x00000001);
    WR_MEM_32(PINCNTL256, 0x00000001);
    WR_MEM_32(PINCNTL257, 0x00000001);
    WR_MEM_32(PINCNTL258, 0x00000001);

    /* PIN MUX for EMAC1 */
    WR_MEM_32(PINCNTL204, 0x000C0002);
    WR_MEM_32(PINCNTL205, 0x000C0002);
    WR_MEM_32(PINCNTL206, 0x000C0002);
    WR_MEM_32(PINCNTL207, 0x000C0002);
    WR_MEM_32(PINCNTL208, 0x00040002);
    WR_MEM_32(PINCNTL209, 0x00040002);
    WR_MEM_32(PINCNTL210, 0x00040002);
    WR_MEM_32(PINCNTL211, 0x00040002);
    WR_MEM_32(PINCNTL212, 0x00040002);
    WR_MEM_32(PINCNTL213, 0x00040002);
    WR_MEM_32(PINCNTL214, 0x00040002);
    WR_MEM_32(PINCNTL215, 0x00040002);
    WR_MEM_32(PINCNTL216, 0x00040002);
    WR_MEM_32(PINCNTL217, 0x00040002);
    WR_MEM_32(PINCNTL218, 0x00000002);
    WR_MEM_32(PINCNTL219, 0x00000002);
    WR_MEM_32(PINCNTL220, 0x00000002);
    WR_MEM_32(PINCNTL221, 0x00000002);
    WR_MEM_32(PINCNTL222, 0x00000002);
    WR_MEM_32(PINCNTL223, 0x00000002);
    WR_MEM_32(PINCNTL224, 0x00000002);
    WR_MEM_32(PINCNTL225, 0x00000002);
    WR_MEM_32(PINCNTL226, 0x00000002);
    WR_MEM_32(PINCNTL227, 0x00000002);

    Uart_stringSend("\t ***** GMII pin mux and Clk initialized....***** \n");
}
STATUS WLAN_Init()
{
	//PLL_CLOCKS_Config();
	//GPIO0ClkEnable(); // called separately
	Uart_stringSend("\r\nSetting Up Functional Clocks for wlan\r\n");

	*(VUINT32*)PINCNTL8 = (UINT32)0x00050001;
	delay(10000);
	*(VUINT32*)PINCNTL9 = (UINT32)0x00060001;
	delay(10000);
	*(VUINT32*)PINCNTL10 = (UINT32)0x00060001;
	delay(10000);
	*(VUINT32*)PINCNTL11 = (UINT32)0x00060001;
	delay(10000);
	*(VUINT32*)PINCNTL12 = (UINT32)0x00060001;
	delay(10000);
	*(VUINT32*)PINCNTL13 = (UINT32)0x00060001;
	delay(10000);
	*(VUINT32*)PINCNTL57 = (UINT32)0x00000080;
	delay(10000);
	*(VUINT32 *)CM_ALWON_GPIO_0_CLKCTRL = (UINT32)0x2;
	delay(10000);
	*(VUINT32 *)CM_ALWON_WLAN_HS_0_CLKSTCTRL = (UINT32)0x2;
	delay(10000);
#if 0
	/* SD/SDIO Pins */
	WR_MEM_32(PINCNTL8,  0x00050001);
	WR_MEM_32(PINCNTL9,  0x00060001);
	WR_MEM_32(PINCNTL10, 0x00060001);
	WR_MEM_32(PINCNTL11, 0x00060001);
	WR_MEM_32(PINCNTL12, 0x00060001);
	WR_MEM_32(PINCNTL13, 0x00060001);
	WR_MEM_32(PINCNTL57, 0x00000080); /* WLAN EN */
#endif

//	WR_MEM_32(CM_ALWON_GPIO_0_CLKCTRL,    0x2); 		/* Enable GPI1 System and Functional Clock*/
	while(RD_MEM_32(CM_ALWON_GPIO_0_CLKCTRL)!=0x2);		/* Poll till Module is functional*/

//	WR_MEM_32(CM_ALWON_WLAN_HS_0_CLKSTCTRL,    0x2); /* SD0 Functional Clock Enable */
	while(RD_MEM_32(CM_ALWON_WLAN_HS_0_CLKSTCTRL)!=0x2);	/*Poll for Module is functional*/

	Uart_stringSend("\r\nWlan Functional Clocks setup\r\n");

}

#endif
/***
 * \brief : UART initialization function for the serial terminal.
 *
 * \b \Desc: This function initialzes the UART 0 with baud rate
 * \v        115200.
 *
 * \param : VOID
 * \return: s32status  SUCCESS  successfully initialized
 * \n                  FAILED   failed to initialized
 */
STATUS UART_Init
(
	void
)
{
	SINT32 s32status;

    /* UART clock initialization*/
	/*Enable the Power Domain Transition of L3 Slow Domain Peripheral*/
	WR_MEM_32(CM_ALWON_L3_SLOW_CLKSTCTRL, POWER_DOMAIN);

	WR_MEM_32(CM_ALWON_UART_0_CLKCTRL, MODULEMODE_ENABLE);
	while(RD_MEM_32(CM_ALWON_UART_0_CLKCTRL)!=0x2);

	while(RD_MEM_32(CM_ALWON_L3_SLOW_CLKSTCTRL) & POWER_DOMAIN_ENABLE
			                                    != POWER_DOMAIN_ENABLE);

	s32status = FAILED;
	Hanhler_uart0 = INIT_VAL;
	/* setting the timeout here */
	uart_timeout = (UINT32)TIME_OUT;

	/**/
	Hanhler_uart0 = _UART_open( UART_0, /*115200*/ 26 );

	if (Hanhler_uart0 )
	{
		s32status = SUCCESS;
	}
    return (s32status);
}
#if 0
STATUS USB_Init()
{

	/* Pin muxing for USB */
	//int CLKIN = 20;

	PLL_Clocks_Config(USB_PLL_BASE,CLKIN,19,960,5,ADPLLJ_CLKCRTL_HS2 + ADPLLJ_CLKCRTL_CLKDCO);

    WR_MEM_32(PINCNTL270, MUX_MODE_RSV_PLLTYPE);
    WR_MEM_32(PINCNTL14, USB_DRVVBUS);
#ifndef disable
    WR_MEM_32(0x48140620, 0x3C1E64F4); /* PHY INIT */
    WR_MEM_32(0x48140628, 0x3C1E64F4); /* PHY INIT */
#endif

	/* USB clock enable */
    WR_MEM_32(CM_ALWON2_USB_CLKCTRL, USB_CLK_EN);
    while((RD_MEM_32(CM_ALWON2_USB_CLKCTRL) & 0x0F) != USB_CLK_EN);  /*Poll for Module is functional*/

    /* Disable local reset */
	WR_MEM_32(RM_ALWON2_RSTCTRL, DISABLE_LOCAL_RESET);



	return(SUCCESS);
}

STATUS LED_PIN_MUX()
{
	UINT32 state;
  *(VUINT32 *)CM_ALWON_GPIO_1_CLKCTRL = (UINT32)GPIO_EN;
   while(*(VUINT32 *)(CM_ALWON_GPIO_1_CLKCTRL)!= GPIO_EN);
   *(VUINT32 *)CM_ALWON_GPIO_0_CLKCTRL = (UINT32)GPIO_EN;
   while(*(VUINT32 *)(CM_ALWON_GPIO_0_CLKCTRL)!= GPIO_EN);

	*(VUINT32 *)MLBP_SIG_IO_CTRL = (UINT32)(0x00000005);
	*((VUINT32 *)PINCNTL15) = (UINT32)GPIO_SEL;
	while(*((VUINT32 *)PINCNTL15) != (UINT32)GPIO_SEL);
	*(VUINT32 *)PINCNTL62 = (UINT32)GPIO_SEL;
	while(*((VUINT32 *)PINCNTL62) != (UINT32)GPIO_SEL);
	state = *(VUINT32 *)PINCNTL15 ;
	*(VUINT32 *)PINCNTL7  = (UINT32)GPIO_SEL;
	*(VUINT32 *)PINCNTL132 = (UINT32)GPIO_SEL;
	*(VUINT32 *)PINCNTL110 = (UINT32)GPIO_SEL;
#if 0
	  WR_MEM_32(MLBP_SIG_IO_CTRL,0x00000005);
	  WR_MEM_32(PINCNTL15, GPIO_SEL);
	  WR_MEM_32(PINCNTL62, GPIO_SEL);
	  WR_MEM_32(PINCNTL7,  GPIO_SEL);
//	  WR_MEM_32(PINCNTL68, GPIO_SEL);
	  WR_MEM_32(PINCNTL132, GPIO_SEL);
	  WR_MEM_32(PINCNTL110, GPIO_SEL);
//	  WR_MEM_32(PINCNTL63, GPIO_SEL);
//	  WR_MEM_32(PINCNTL64, GPIO_SEL);
//	  WR_MEM_32(PINCNTL65, GPIO_SEL);
#endif
//	   GPIO0ClkEnable();
//	   GPIO1ClkEnable();
	   *(VUINT32 *)CM_ALWON_GPIO_1_CLKCTRL = (UINT32)GPIO_EN;
	   while(*(VUINT32 *)(CM_ALWON_GPIO_1_CLKCTRL)!= GPIO_EN);
	   *(VUINT32 *)CM_ALWON_GPIO_0_CLKCTRL = (UINT32)GPIO_EN;
	   while(*(VUINT32 *)(CM_ALWON_GPIO_0_CLKCTRL)!= GPIO_EN);
	   printf("\n PIN muxing done\n");
}


STATUS AudioInit()
{

	    Uart_stringSend("\r\n****  DM8127 ALL ADPLL INIT IS In Progress ......... \n");
	    PLL_Clocks_Config(USB_PLL_BASE,CLKIN,19,800,4,ADPLLJ_CLKCRTL_HS2 + ADPLLJ_CLKCRTL_CLKDCO);
//		PLL_SETUP();cmdAUDIOPLL(CLKIN,19,800,4);
		WR_MEM_32(0x48180324, 0x03);
		Uart_stringSend("\t ****  DM8127 ALL ADPLL INIT IS  Done ************** \n");
	    /* DM8127 change: Changed to MCA1 instead of MCA2 */
	    WR_MEM_32(PINCNTL16, 0x00000004);  /* MCA2-AHCLKX */
	    WR_MEM_32(PINCNTL39, 0x00000001);  /* MCA2-ACLKX */
	    WR_MEM_32(PINCNTL40, 0x00000001);  /* MCA2-AFSX */
	    WR_MEM_32(PINCNTL41, 0x00000001);  /* MCA2-AXR0 */
	    WR_MEM_32(PINCNTL42, 0x00000001);  /* MCA2-AXR1 */
		WR_MEM_32(PINCNTL43, 0x00000001);  /* MCA2-AXR2 */
		WR_MEM_32(PINCNTL44, 0x00000001);  /* MCA2-AXR3 */
	    WR_MEM_32(PINCNTL135, 0x000E0040);  /* I2c2-sda */
		WR_MEM_32(PINCNTL136, 0x000E0040);  /*  I2c2-scl */

		I2C0Enable();
		I2C2Enable();

	    WR_MEM_32(CM_ALWON_MCA_2_CLKSTCTRL,    0x2); /*Enable McASP2 Clock*/
	    while(RD_MEM_32(CM_ALWON_MCA_2_CLKSTCTRL)!=0x2);        /*Poll for Module is functional*/


}
#endif

/***
 * \brief:    String send funtion to the UART 0 (serial terminal).
 *
 * \b \Descr: This function send a string to the serial terminal.
 *
 * \param : const char*  [IN]  string that has to send on the serial terminal.
 * \return: VOID
 */
void Uart_stringSend
(
	const char* pu8string
)
{
	UINT32 u32len;
	UINT32 u32count;

	u32len = strlen((const char *)pu8string);
	for (u32count = 0; u32count < u32len; u32count ++)
	{
		uart_send( Hanhler_uart0, pu8string[u32count], uart_timeout );
	}
}

/***
 * \brief : Uart print fumnction for the formate specifier.
 *
 * \b \Descr : This function takes the parameters as formate specifiers as well
 * \n          as the parameter value which has to be print. fially the data
 * \n          is sent on the serial terminal.
 *
 * \param :   const char *  [IN]  formate specifier and data that has to be print
 * \return:   SINT32        Number of character that sent to the UART
 */
SINT32 platform_write
(
	const char *format, ...
)
{
	va_list arg_ptr;
	UINT32	u32count;
	UINT32  length;

	/* sanity check */
	if (strlen(format) > MAX_LEN)
	{
#ifdef DEBG
		printf("\r\n Input error \r\n");
#endif
		 return -1;
	}

#ifdef NO_LONGER_VALID
	/* Open Uart Handle */
	if (( uart0 = _UART_open( UART_0, /*115200*/ 26 ) ) == 0 )
	{
#ifdef DEBG
	  printf("\r\n failed to initialize the error \r\n");
#endif
	  	  return -1;
	}

=======
#endif

	va_start( arg_ptr, format );

	length = vsprintf( (char *)write_buffer, format, arg_ptr );
	va_end( arg_ptr );

	if (write_type == PLATFORM_WRITE_UART)
	{
		for (u32count = 0; u32count < length; u32count ++)
		{
			if (write_buffer[u32count] == '\n') {
				uart_send(Hanhler_uart0, (unsigned char)0x0D, uart_timeout);
				uart_send(Hanhler_uart0, (unsigned char)0x0A, uart_timeout);
			}
			else {
				uart_send(Hanhler_uart0, (unsigned char)write_buffer[u32count], uart_timeout);
			}
		}
	}
	else if (write_type == PLATFORM_WRITE_PRINTF)
	{
		/*Used to print on the CCS console*/
		printf("%s", write_buffer);
		fflush(stdout);
	}
	return length;
}


/**
 * \brief : Uart receive fumnction.
 *
 * \b \Descr : This function read the data/ value from the UART0
 * \n
 * \n
 *
 * \param :   *data    [OUT]  formate specifier and data that has to be print
 * \return:   length   [IN]   Number of character that sent to the UART
 */
SINT32 platform_read
(
	char *data,
	UINT32 length
)
{
	SINT32 status = 0;
	UINT32 count = 0;
	UART_Handle uart0;

	/* Open Uart Handle */
		    if ( ( uart0 = _UART_open( UART_0, /*115200*/ 26 ) ) == 0 )
		        return -1;

	if (read_type == PLATFORM_READ_UART)
	{
		/* Wait till an input is provided */
		do
		{
		status = uart_recv (uart0, data, uart_timeout);//
		if(status == 0)
		{
			/* Check if user pressed enter */
			if((*data == 0xD) || (*data == 0x1B))
			{
			   *data = '\0';
			   break;
			}

			uart_send(uart0, *data, uart_timeout);

			count++;
			data++;
		}
		else
		{
			platform_write("UART read error");
			return -1;
		}
		} while ((status != 0) || (count != length));
	}

	if (read_type == PLATFORM_READ_SCANF)
		{
			count = scanf("%s", data);
		}

return (count);
}



#ifdef NO_LONGER_VALID
SINT32 Platform_init
(
	void
)
{
	SINT32 s32status;
    UINT32 u32strlen;
    UINT32 u32count;
    UINT8 u8message[] = "\r\nUart 0 initialized successfully\r\n";
	UINT8 u8ErrorMessage[] = "\r\nUart 0 failed to initialize\r\n";

	s32status = FAILED;
	CLKIN = 20;
/***************************** UART INITIALIZATION ********************/

	s32status = UART_Init();
	if(SUCCESS == s32status)
	{
		u32strlen = strlen((const char *)u8message);
		for(u32count = 0 ; u32count < u32strlen; u32count ++)
		{
			uart_send( Hanhler_uart0, u8message[u32count], uart_timeout );
		}
	}
	else
	{
		u32strlen = strlen((const char *)u8ErrorMessage);
		for(u32count = 0 ; u32count < u32strlen; u32count ++)
		{
			uart_send( Hanhler_uart0, u8ErrorMessage[u32count], uart_timeout );
		}
	}
/***************************** USB Initialization *********************/
#ifdef USB_INIT
	s32status = USB_Init();
#endif

/***************************** Audio codec Initialization *********************/
#if 0
	Uart_stringSend("\t ====== Initializes the DM8127 board for AIC3104 test ===== \n");
	s32status = AudioInit();
#endif

#ifdef DEMO_TEST
	s32status = LED_PIN_MUX();
	Uart_stringSend((const UINT8 *)u8ErrorMessage);
#endif
	return(s32status);

}
#endif
